<?php
/**
 * 药品数据库查询系统 - 智能组合搜索版
 * 使用说明：
 * 1. 将此文件保存为 index.php
 * 2. 将 drugs.db 放在同一目录
 * 3. 支持空格分隔多关键词组合搜索
 */
//刀客源码网www.dkewl.com
// 数据库配置
define('DB_PATH', __DIR__ . '/drugs.db');
 
// 错误处理
error_reporting(E_ALL);
ini_set('display_errors', 0);
 
// 处理查询请求
$results = [];
$error = '';
$query_info = '';
$search_term = '';
$search_type = 'smart';
 
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $search_term = trim($_POST['search_term'] ?? '');
    $search_type = $_POST['search_type'] ?? 'smart';
     
    if (!empty($search_term)) {
        try {
            $pdo = new PDO('sqlite:' . DB_PATH);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
             
            // 智能搜索：支持空格分隔的多关键词组合
            $keywords = array_filter(explode(' ', $search_term));
             
            if ($search_type === 'smart' && count($keywords) > 1) {
                // 多关键词组合搜索：所有关键词都必须在某个字段中出现
                $conditions = [];
                $params = [];
                 
                foreach ($keywords as $idx => $keyword) {
                    $param_name = "keyword{$idx}";
                    $conditions[] = "(通用名称 LIKE :{$param_name} OR 
                                     商品名称 LIKE :{$param_name} OR 
                                     生产企业 LIKE :{$param_name} OR 
                                     批准文号 LIKE :{$param_name} OR 
                                     相关疾病 LIKE :{$param_name} OR 
                                     适应症 LIKE :{$param_name})";
                    $params[$param_name] = "%{$keyword}%";
                }
                 
                $sql = "SELECT * FROM drug_info WHERE " . implode(' AND ', $conditions);
                $sql .= " ORDER BY 通用名称, 生产企业 LIMIT 100";
                 
                $query_info = "智能组合搜索（" . count($keywords) . " 个关键词）：\"" . $search_term . "\"";
                 
            } else {
                // 单关键词或指定字段搜索
                $keyword = $keywords[0];
                 
                switch ($search_type) {
                    case 'generic_name':
                        $sql = "SELECT * FROM drug_info WHERE 通用名称 LIKE :term ORDER BY 通用名称, 生产企业";
                        $query_info = "通用名称";
                        break;
                    case 'trade_name':
                        $sql = "SELECT * FROM drug_info WHERE 商品名称 LIKE :term ORDER BY 商品名称, 生产企业";
                        $query_info = "商品名称";
                        break;
                    case 'manufacturer':
                        $sql = "SELECT * FROM drug_info WHERE 生产企业 LIKE :term ORDER BY 生产企业, 通用名称";
                        $query_info = "生产企业";
                        break;
                    case 'approval_number':
                        $sql = "SELECT * FROM drug_info WHERE 批准文号 LIKE :term ORDER BY 批准文号";
                        $query_info = "批准文号";
                        break;
                    default:
                        // 智能模式单关键词：搜索所有主要字段
                        $sql = "SELECT * FROM drug_info WHERE 
                                通用名称 LIKE :term OR 
                                商品名称 LIKE :term OR 
                                生产企业 LIKE :term OR 
                                批准文号 LIKE :term OR 
                                相关疾病 LIKE :term OR 
                                适应症 LIKE :term 
                                ORDER BY 通用名称, 生产企业";
                        $query_info = "智能搜索";
                }
                $sql .= " LIMIT 100";
                $params = ['term' => "%{$keyword}%"];
                $query_info .= " 包含 \"{$keyword}\"";
            }
             
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $results = $stmt->fetchAll();
             
            $query_info .= " 的药品，共找到 " . count($results) . " 条结果";
             
        } catch (PDOException $e) {
            $error = "数据库查询失败: " . $e->getMessage();
        }
    } else {
        $error = "请输入查询关键词";
    }
}
 
// 获取数据库统计信息
$db_stats = '';
try {
    $pdo = new PDO('sqlite:' . DB_PATH);
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM drug_info");
    $total = $stmt->fetch()['total'];
    $db_stats = "数据库共收录 {$total} 条药品信息";
} catch (PDOException $e) {
    $db_stats = "无法连接数据库";
}
 
// 辅助函数：安全输出
function safe_output($value) {
    return !empty($value) && $value !== '-' ? nl2br(htmlspecialchars($value)) : '<span style="color: #bbb;">暂无</span>';
}
?>
 
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>药品数据库查询系统</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
         
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", "Microsoft YaHei", sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
         
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
         
        .header {
            background: white;
            border-radius: 10px;
            padding: 25px 30px;
            margin-bottom: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
         
        h1 {
            color: #667eea;
            margin-bottom: 8px;
            font-size: 26px;
        }
         
        .db-stats {
            color: #666;
            font-size: 14px;
        }
         
        .search-box {
            background: white;
            border-radius: 10px;
            padding: 25px 30px;
            margin-bottom: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
         
        .search-hint {
            color: #667eea;
            font-size: 13px;
            margin-bottom: 15px;
            padding: 10px;
            background: #f0f4ff;
            border-radius: 5px;
            border-left: 3px solid #667eea;
        }
         
        .search-form {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
         
        .search-type {
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 5px;
            font-size: 15px;
            flex: 0 0 140px;
            background: white;
        }
         
        .search-input {
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 5px;
            font-size: 15px;
            flex: 1;
            min-width: 250px;
        }
         
        .search-input:focus {
            outline: none;
            border-color: #667eea;
        }
         
        .search-btn {
            padding: 12px 30px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 5px;
            font-size: 15px;
            cursor: pointer;
            transition: background 0.3s;
            font-weight: 500;
        }
         
        .search-btn:hover {
            background: #5568d3;
        }
         
        .message {
            background: white;
            border-radius: 10px;
            padding: 15px 20px;
            margin-bottom: 20px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
         
        .error {
            background: #fee;
            color: #c33;
            border-left: 4px solid #c33;
        }
         
        .info {
            background: #e7f3ff;
            color: #0066cc;
            border-left: 4px solid #0066cc;
        }
         
        .results {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
         
        .drug-card {
            border: 1px solid #ddd;
            border-radius: 8px;
            margin-bottom: 20px;
            overflow: hidden;
            transition: all 0.3s;
            background: #fafafa;
        }
         
        .drug-card:hover {
            border-color: #667eea;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.15);
        }
         
        /* 核心信息区 */
        .drug-header {
            background: white;
            padding: 20px 25px;
            border-bottom: 2px solid #f0f0f0;
        }
         
        .drug-title-row {
            display: flex;
            align-items: baseline;
            gap: 15px;
            margin-bottom: 15px;
            flex-wrap: wrap;
        }
         
        .drug-generic-name {
            font-size: 22px;
            font-weight: bold;
            color: #333;
        }
         
        .drug-trade-name {
            font-size: 16px;
            color: #667eea;
            font-weight: 500;
        }
         
        .drug-badges {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
         
        .badge {
            display: inline-block;
            padding: 4px 12px;
            background: #f0f0f0;
            border-radius: 4px;
            font-size: 12px;
            color: #666;
        }
         
        .badge.primary {
            background: #e7f3ff;
            color: #0066cc;
        }
         
        /* 基础信息网格 */
        .basic-info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 12px;
            padding: 20px 25px;
            background: white;
        }
         
        .info-row {
            display: flex;
            gap: 10px;
            font-size: 14px;
            line-height: 1.6;
        }
         
        .info-label {
            font-weight: 600;
            color: #555;
            min-width: 80px;
            flex-shrink: 0;
        }
         
        .info-value {
            color: #333;
            flex: 1;
        }
         
        /* 详细说明区 */
        .drug-details {
            padding: 20px 25px;
            background: white;
        }
         
        .detail-block {
            margin-bottom: 20px;
            padding-bottom: 20px;
            border-bottom: 1px solid #f0f0f0;
        }
         
        .detail-block:last-child {
            border-bottom: none;
            margin-bottom: 0;
            padding-bottom: 0;
        }
         
        .detail-title {
            font-weight: bold;
            color: #667eea;
            margin-bottom: 8px;
            font-size: 15px;
        }
         
        .detail-content {
            color: #555;
            line-height: 1.8;
            font-size: 14px;
        }
         
        .section-divider {
            height: 2px;
            background: #f0f0f0;
            margin: 0;
        }
         
        .no-results {
            text-align: center;
            padding: 40px;
            color: #999;
            font-size: 16px;
        }
         
        @media (max-width: 768px) {
            .basic-info-grid {
                grid-template-columns: 1fr;
            }
        }
         
        @media print {
            body {
                background: white;
            }
            .search-box, .header {
                display: none;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>药品数据库查询系统</h1>
            <div class="db-stats"><?php echo htmlspecialchars($db_stats); ?></div>
        </div>
         
        <div class="search-box">
            <div class="search-hint">
                &#128161; <strong>智能搜索提示：</strong>输入多个关键词用空格分隔，如 "阿莫西林 进口"、"布洛芬 儿童"，系统将自动组合匹配
            </div>
            <form method="POST" class="search-form">
                <select name="search_type" class="search-type">
                    <option value="smart" <?php echo $search_type === 'smart' ? 'selected' : ''; ?>>智能搜索</option>
                    <option value="generic_name" <?php echo $search_type === 'generic_name' ? 'selected' : ''; ?>>通用名称</option>
                    <option value="trade_name" <?php echo $search_type === 'trade_name' ? 'selected' : ''; ?>>商品名称</option>
                    <option value="manufacturer" <?php echo $search_type === 'manufacturer' ? 'selected' : ''; ?>>生产企业</option>
                    <option value="approval_number" <?php echo $search_type === 'approval_number' ? 'selected' : ''; ?>>批准文号</option>
                </select>
                <input type="text" name="search_term" class="search-input"
                       placeholder="如：阿莫西林 进口"
                       value="<?php echo htmlspecialchars($search_term); ?>" required>
                <button type="submit" class="search-btn">&#128269; 查询</button>
            </form>
        </div>
         
        <?php if ($error): ?>
            <div class="message error">
                &#10060; <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>
         
        <?php if ($query_info): ?>
            <div class="message info">
                &#8505;&#65039; <?php echo htmlspecialchars($query_info); ?>
            </div>
        <?php endif; ?>
         
        <?php if (!empty($results)): ?>
            <div class="results">
                <?php foreach ($results as $drug): ?>
                    <div class="drug-card">
                        <!-- 核心信息区 -->
                        <div class="drug-header">
                            <div class="drug-title-row">
                                <span class="drug-generic-name">
                                    <?php echo htmlspecialchars($drug['通用名称'] ?? '未知药品'); ?>
                                </span>
                                <?php if (!empty($drug['商品名称'])): ?>
                                    <span class="drug-trade-name">
                                        [<?php echo htmlspecialchars($drug['商品名称']); ?>]
                                    </span>
                                <?php endif; ?>
                            </div>
                             
                            <div class="drug-badges">
                                <?php if (!empty($drug['药品性质'])): ?>
                                    <span class="badge primary"><?php echo htmlspecialchars($drug['药品性质']); ?></span>
                                <?php endif; ?>
                                <?php if (!empty($drug['药品分类'])): ?>
                                    <span class="badge"><?php echo htmlspecialchars($drug['药品分类']); ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                         
                        <!-- 基础信息网格 -->
                        <div class="basic-info-grid">
                            <div class="info-row">
                                <span class="info-label">生产企业</span>
                                <span class="info-value"><?php echo safe_output($drug['生产企业']); ?></span>
                            </div>
                             
                            <div class="info-row">
                                <span class="info-label">批准文号</span>
                                <span class="info-value"><?php echo safe_output($drug['批准文号']); ?></span>
                            </div>
                             
                            <div class="info-row">
                                <span class="info-label">规格</span>
                                <span class="info-value"><?php echo safe_output($drug['规格']); ?></span>
                            </div>
                             
                            <div class="info-row">
                                <span class="info-label">汉语拼音</span>
                                <span class="info-value"><?php echo safe_output($drug['汉语拼音']); ?></span>
                            </div>
                             
                            <div class="info-row">
                                <span class="info-label">贮藏</span>
                                <span class="info-value"><?php echo safe_output($drug['贮藏']); ?></span>
                            </div>
                             
                            <div class="info-row">
                                <span class="info-label">有效期</span>
                                <span class="info-value"><?php echo safe_output($drug['有效期']); ?></span>
                            </div>
                             
                            <?php if (!empty($drug['相关疾病'])): ?>
                            <div class="info-row">
                                <span class="info-label">相关疾病</span>
                                <span class="info-value"><?php echo safe_output($drug['相关疾病']); ?></span>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['性状'])): ?>
                            <div class="info-row">
                                <span class="info-label">性状</span>
                                <span class="info-value"><?php echo safe_output($drug['性状']); ?></span>
                            </div>
                            <?php endif; ?>
                        </div>
                         
                        <div class="section-divider"></div>
                         
                        <!-- 详细说明区 -->
                        <div class="drug-details">
                            <?php if (!empty($drug['主要成份'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">主要成份</div>
                                <div class="detail-content"><?php echo safe_output($drug['主要成份']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['适应症'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">适应症</div>
                                <div class="detail-content"><?php echo safe_output($drug['适应症']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['用法用量'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">用法用量</div>
                                <div class="detail-content"><?php echo safe_output($drug['用法用量']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['不良反应'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">不良反应</div>
                                <div class="detail-content"><?php echo safe_output($drug['不良反应']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['禁忌'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">禁忌</div>
                                <div class="detail-content"><?php echo safe_output($drug['禁忌']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['注意事项'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">注意事项</div>
                                <div class="detail-content"><?php echo safe_output($drug['注意事项']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['孕妇及哺乳期妇女用药'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">孕妇及哺乳期妇女用药</div>
                                <div class="detail-content"><?php echo safe_output($drug['孕妇及哺乳期妇女用药']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['儿童用药'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">儿童用药</div>
                                <div class="detail-content"><?php echo safe_output($drug['儿童用药']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['老人用药'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">老人用药</div>
                                <div class="detail-content"><?php echo safe_output($drug['老人用药']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['药物相互作用'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">药物相互作用</div>
                                <div class="detail-content"><?php echo safe_output($drug['药物相互作用']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['药理毒理'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">药理毒理</div>
                                <div class="detail-content"><?php echo safe_output($drug['药理毒理']); ?></div>
                            </div>
                            <?php endif; ?>
                             
                            <?php if (!empty($drug['药代动力学'])): ?>
                            <div class="detail-block">
                                <div class="detail-title">药代动力学</div>
                                <div class="detail-content"><?php echo safe_output($drug['药代动力学']); ?></div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($error)): ?>
            <div class="results">
                <div class="no-results">&#128533; 未找到符合条件的药品信息</div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>